using System;
using System.Data;
using System.Collections;
using System.Windows.Forms;
using Method = System.Reflection.MethodBase;
using System.Drawing.Printing;
using gov.va.med.vbecs.reports;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Text;
using System.Xml;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using Microsoft.Reporting.WinForms;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Contains methods and properties used by VBECS reports
	/// </summary>
	[Serializable]
	public abstract class VbecsReport:BaseBusinessObject//,IReport
	{
		private DataSet _parametersDataSet;
		//
		private DataRow _parameterRow;
		//
		private string _serialFilePath;

		private bool _isScheduled = false;

		#region Constructors

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5344"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>VbecsReport object initialized with start and end dates defaulted to previous day</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5345"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty VBECS report constructor. Defaults start and end date to previous day.
		/// </summary>
		public VbecsReport()
		{
			//
			//TODO: Need additional filename string if file exists, don't overwrite.
			_serialFilePath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData) + "\\" + this.GetType().Name + DateTime.UtcNow.ToString("yyyyMMddHHmmss") + "_Report.xml"; //GetEnvironmentVariable("TEMP") 
			//
			string pdsName = this.GetType().Name;
			//
			_parametersDataSet = new DataSet(pdsName);
			_parametersDataSet.Tables.Add(new DataTable(ARTIFICIAL.ReportParameters));
			DataTable dt = _parametersDataSet.Tables[ARTIFICIAL.ReportParameters];
			//
			PropertyInfo[] propInfoArray = this.GetType().GetProperties();
			//
			foreach(PropertyInfo propInfo in propInfoArray)
			{
				MethodInfo meth = propInfo.GetSetMethod();
				if (meth != null)
				{
					Debug.WriteLine(propInfo.Name);
					Debug.WriteLine(propInfo.PropertyType.Name);
					dt.Columns.Add(propInfo.Name, propInfo.PropertyType);
				}
				else
				{
					Debug.WriteLine(propInfo.Name);
					Debug.WriteLine(propInfo.PropertyType.Name);
				}
			}
			_parameterRow = dt.NewRow();
			dt.Rows.Add(_parameterRow);
			//ParameterTypeName = this.GetType().FullName;
			//
			if (this.DivisionCode == null)
			{
				this.DivisionCode = Common.LogonUser.LogonUserDivisionCode;
			}
			//
			
			// Setting a default for the sheduling capabilities.
			this.PrintDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			//BR_77.01
			this.StartDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime(this.DivisionCode).Date.AddDays(-1);
			this.EndDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime(this.DivisionCode).Date.AddDays(-1);
			//
			this.IsMultiDivisional = false;
			this.IsScheduled = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5346"> 
		///		<ExpectedInput>Valid start and end dates, multi-divisional flag set to true, scheduled flag set to true</ExpectedInput>
		///		<ExpectedOutput>VbecsReport object initialized</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5347"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS report constructor.
		/// </summary>
		/// <param name="startDate">Report start date</param>
		/// <param name="endDate">Reoprt end date</param>
		/// <param name="isMultiDivisional">Is report multidivisional</param>
		/// <param name="isScheduled">Is report schedule for printing</param>
		public VbecsReport(DateTime startDate, DateTime endDate, bool isMultiDivisional, bool isScheduled):this()
		{
			this.StartDate = startDate;
			this.EndDate = endDate;
			//
			this.IsMultiDivisional = isMultiDivisional;
			this.IsScheduled = isScheduled;
			this.PrintDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
		}

		#endregion

		#region Properties



		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6546"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6547"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets parameter row
		/// </summary>
		public DataRow ParameterRow
		{
			get
			{
				return _parameterRow;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6548"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6549"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Report Name
		/// </summary>
		public string ReportName
		{
			get
			{
				return ParameterRow[ARTIFICIAL.ReportName].ToString();
			}
			set
			{
				ParameterRow[ARTIFICIAL.ReportName] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6550"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6551"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Division Code
		/// </summary>
		public string DivisionCode
		{
			get
			{
				return ParameterRow[TABLE.VamcDivision.DivisionCode].ToString();
			}
			set
			{
				ParameterRow[TABLE.VamcDivision.DivisionCode] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6552"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6553"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets serial file path
		/// </summary>
		public string SerialFilePath
		{
			get
			{
				return _serialFilePath;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5348"> 
		///		<ExpectedInput>Valid VbecsReport object</ExpectedInput>
		///		<ExpectedOutput>StartDate property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5349"> 
		///		<ExpectedInput>Valid start date (less than End Date)</ExpectedInput>
		///		<ExpectedOutput>VbecsReport object updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5350"> 
		///		<ExpectedInput>Invalid start date (greater than End Date)</ExpectedInput>
		///		<ExpectedOutput>Rule Broken</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets report date range - start date.
		/// </summary>
		public DateTime StartDate
		{
			get
			{
				if (this._parameterRow.IsNull(ARTIFICIAL.StartDate))
				{
					return DateTime.MinValue;
				}
				return Convert.ToDateTime(this._parameterRow[ARTIFICIAL.StartDate]);
			}
			set
			{
				//BR_77.05
				this._parameterRow[ARTIFICIAL.StartDate] = value.Date;
				//
				RuleBroken(ARTIFICIAL.EndDate, (this.EndDate < this.StartDate));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5351"> 
		///		<ExpectedInput>Valid VbecsReport object</ExpectedInput>
		///		<ExpectedOutput>EndDate property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5352"> 
		///		<ExpectedInput>Valid end date ( greater than Start Date)</ExpectedInput>
		///		<ExpectedOutput>VbecsReport object updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5353"> 
		///		<ExpectedInput>Invalid end date (less than Start Date)</ExpectedInput>
		///		<ExpectedOutput>Rule Broken</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5354"> 
		///		<ExpectedInput>Invalid end date ( greater than Print Date)</ExpectedInput>
		///		<ExpectedOutput>Rule Broken</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets report date range - end date.
		/// </summary>
		public DateTime EndDate
		{
			get
			{
				if (this._parameterRow.IsNull(ARTIFICIAL.EndDate))
				{
					return DateTime.MinValue;
				}
				return Convert.ToDateTime(this._parameterRow[ARTIFICIAL.EndDate]);
			}
			set
			{
				//BR_77.05
				this._parameterRow[ARTIFICIAL.EndDate] = value.Date.AddHours(23).AddMinutes(59).AddSeconds(59);
				//
				RuleBroken(Method.GetCurrentMethod().Name, (this.EndDate < this.StartDate));
				//BR_77.11
				RuleBroken(ARTIFICIAL.PrintDate, (this._isScheduled && ((this.PrintDate < BOL.VBECSDateTime.GetDivisionCurrentDateTime())||(this.PrintDate < this.EndDate))));

			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5355"> 
		///		<ExpectedInput>Valid VbecsReport object with scheduled flag set to true</ExpectedInput>
		///		<ExpectedOutput>PrintDate property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5356"> 
		///		<ExpectedInput>Valid VbecsReport object with scheduled flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5357"> 
		///		<ExpectedInput>Valid print date ( greater than End Date and current date)</ExpectedInput>
		///		<ExpectedOutput>VbecsReport object updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5358"> 
		///		<ExpectedInput>Invalid print date (less than current date)</ExpectedInput>
		///		<ExpectedOutput>Rule Broken</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets scheduled report print date. (Scheduled reports only).
		/// </summary>
		public virtual DateTime PrintDate
		{
			get
			{
				if (!this._isScheduled)
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.ReportNotScheduled().ResString);
				}
				return Convert.ToDateTime(this._parameterRow[ARTIFICIAL.PrintDate]);
				//this._printDate;
			}
			set
			{
				this._parameterRow[ARTIFICIAL.PrintDate]=value; 
				//this._printDate = value;
				//BR_77.11
				RuleBroken(Method.GetCurrentMethod().Name, (this._isScheduled && ((Common.VBECSDateTime.CompareDates(this.PrintDate,BOL.VBECSDateTime.GetDivisionCurrentDateTime())<0)||(this.PrintDate < this.EndDate))));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5359"> 
		///		<ExpectedInput>Valid VbecsReport object</ExpectedInput>
		///		<ExpectedOutput>IsScheduled property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5360"> 
		///		<ExpectedInput>Valid IsScheduled flag set to false</ExpectedInput>
		///		<ExpectedOutput>Updated VbecsReport object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5361"> 
		///		<ExpectedInput>IsScheduled flag set to true, with no printer defined</ExpectedInput>
		///		<ExpectedOutput>Rule broken</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether the report is to be scheduled for printing
		/// </summary>
		public bool IsScheduled
		{
			get
			{
				return this._isScheduled;
			}
			set
			{
				this._isScheduled = value;
				//
				if (!this._isScheduled)
				{
					RuleBroken(ARTIFICIAL.PrintDate,false);
					RuleBroken(ARTIFICIAL.PrinterName,false);
				}
				else
				{
					RuleBroken(ARTIFICIAL.PrintDate, ((Common.VBECSDateTime.CompareDates(this.PrintDate,BOL.VBECSDateTime.GetDivisionCurrentDateTime())<0)||(this.PrintDate < this.EndDate)));
					RuleBroken(ARTIFICIAL.PrinterName, ((this.PrinterName == null) || (this.PrinterName.Trim().Length == 0)));
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5362"> 
		///		<ExpectedInput>Valid VbecsReport object</ExpectedInput>
		///		<ExpectedOutput>IsMultiDivisional property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5363"> 
		///		<ExpectedInput>Valid multi divisional flag</ExpectedInput>
		///		<ExpectedOutput>Updated VbecsReport object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5364"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether the report is multi-divisional
		/// </summary>
		public bool IsMultiDivisional
		{
			get
			{
				return Convert.ToBoolean(this._parameterRow[ARTIFICIAL.IsMultiDivisional]);
			}
			set
			{
				this._parameterRow[ARTIFICIAL.IsMultiDivisional] = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5365"> 
		///		<ExpectedInput>Valid VbecsReport object</ExpectedInput>
		///		<ExpectedOutput>IsPreliminary property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5366"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets value indicating whether the report is preliminary (BR_77.02)
		/// </summary>
		public bool IsPreliminary
		{
			get
			{
				return (this.EndDate.Date == BOL.VBECSDateTime.GetDivisionCurrentDateTime().Date);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5367"> 
		///		<ExpectedInput>Valid VbecsReport object with scheduled flag set to true</ExpectedInput>
		///		<ExpectedOutput>PrinterName property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5368"> 
		///		<ExpectedInput>Valid VbecsReport object with scheduled flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5369"> 
		///		<ExpectedInput>Valid printer name</ExpectedInput>
		///		<ExpectedOutput>Updated VbecsReport object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5370"> 
		///		<ExpectedInput>No printer name with scheduled flag set to true</ExpectedInput>
		///		<ExpectedOutput>Rule Broken</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5371"> 
		///		<ExpectedInput>Invalid printer name</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets name of printer. (Scheduled reports only).
		/// </summary>
		public string PrinterName
		{
			get
			{
				if (!this._isScheduled)
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.ReportNotScheduled().ResString);
				}
				return this._parameterRow[ARTIFICIAL.PrinterName].ToString();
			}
			set
			{
				if ((value != null) && (value.Trim().Length>0) && !IsPrinterConfigured(value))
				{
					throw new BusinessObjectException("Selected printer is not configured");
				}
				this._parameterRow[ARTIFICIAL.PrinterName] = value;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((this._isScheduled) && ((this.PrinterName == null) || (this.PrinterName.Trim().Length == 0))));
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6554"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6555"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Printed By
		/// </summary>
		public string PrintedBy
		{
			get
			{
				if (this._parameterRow.IsNull(ARTIFICIAL.PrintedBy))
				{
					return Common.LogonUser.LogonUserName;
				}
				return this._parameterRow[ARTIFICIAL.PrintedBy].ToString();
			}
			set
			{
				this._parameterRow[ARTIFICIAL.PrintedBy] = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5372"> 
		///		<ExpectedInput>Valid VbecsReport object</ExpectedInput>
		///		<ExpectedOutput>IsScheduledReportGreaterThanThirtyDays property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5373"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets value indicating whether the report date range is more than 30 days. (Scheduled reports only). 
		/// BR_77.11
		/// </summary>
		public bool IsScheduledReportGreaterThanThirtyDays
		{
			get
			{
				return this.IsScheduled && (this.StartDate.AddDays(30) < this.EndDate);
			}
		}

		#endregion

		#region BaseBusinessObject

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7965"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7966"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ** Not implemented **
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}

		/// <summary>
		/// ** Not implemented **
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
		}




		#endregion

		#region Static methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5374"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Array list of installed printer names</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5375"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static method returns a list of installed printers
		/// </summary>
		/// <returns>List of installed printers</returns>
		public static ArrayList GetPrinters()
		{
			ArrayList printerList = new ArrayList();
		
			foreach(String printer in PrinterSettings.InstalledPrinters)
			{
				printerList.Add(printer);
			}
			return printerList;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5376"> 
		///		<ExpectedInput>Valid user name</ExpectedInput>
		///		<ExpectedOutput>Data table containing report header info</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5377"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Data Access Layer Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Builds the DataSet used as a source for report header (BR_77.07)
		/// </summary>
		/// <param name="userName">User Name</param>
		/// <returns>Report Header dataset</returns>
		public static DataSet GetReportHeader(string userName)
		{
			return BOL.VbecsReport.GetReportHeader(userName, Common.LogonUser.LogonUserDivisionCode);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5378"> 
		///		<ExpectedInput>Valid user name and division code</ExpectedInput>
		///		<ExpectedOutput>Data table containing report header info</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5379"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Data Access Layer Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Builds the DataSet used as a source for report header (BR_77.07)
		/// </summary>
		/// <param name="userName">User Name</param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns>Report Header dataset</returns>
		public static DataSet GetReportHeader(string userName, string divisionCode)
		{
			//We cannot use Common.LogonUser.LogonUserDivisionCode because a service
			//that is not logged in to Vbecs may be creating the report
			BOL.Division division = new BOL.Division(divisionCode);
			BOL.LocalSupplier localSupplier = new BOL.LocalSupplier(division.LocalSupplierGuid, true);
			
			DataSet dsHeader = new DataSet("VBECSReportHeader");
			DataTable dtHeader = new DataTable("ReportHeader");
			dsHeader.Tables.Add(dtHeader);

			dtHeader.Columns.Add(TABLE.VamcDivision.DivisionCode);
			dtHeader.Columns.Add(TABLE.VamcDivision.DivisionName);
			dtHeader.Columns.Add(ARTIFICIAL.TechId);
			dtHeader.Columns.Add(TABLE.Address.AddressLine1);
			dtHeader.Columns.Add(TABLE.Address.AddressLine2);
			dtHeader.Columns.Add(TABLE.Address.AddressLine3);
			dtHeader.Columns.Add(ARTIFICIAL.City);
			dtHeader.Columns.Add(ARTIFICIAL.State);
			dtHeader.Columns.Add(ARTIFICIAL.Zip);
			dtHeader.Columns.Add(ARTIFICIAL.EnvironmentType);
			dtHeader.Columns.Add(ARTIFICIAL.PrintDate,typeof(DateTime));

			DataRow drHeader = dtHeader.NewRow();
			drHeader[TABLE.VamcDivision.DivisionCode] = division.DivisionCode;
			drHeader[TABLE.VamcDivision.DivisionName] = division.DivisionName;
			drHeader[ARTIFICIAL.TechId] = userName;
			drHeader[TABLE.Address.AddressLine1] = localSupplier.Address.AddressLine1;
			drHeader[TABLE.Address.AddressLine2] = localSupplier.Address.AddressLine2;
			drHeader[TABLE.Address.AddressLine3] = localSupplier.Address.AddressLine3;
			drHeader[ARTIFICIAL.City] = localSupplier.Address.City;
			drHeader[ARTIFICIAL.State] = localSupplier.Address.StateCode;
			drHeader[ARTIFICIAL.Zip] = localSupplier.Address.ZipCode;
			drHeader[ARTIFICIAL.EnvironmentType] = VbecsConfig.Current.EnvironmentType;
			drHeader[ARTIFICIAL.PrintDate] = BOL.VBECSDateTime.GetDivisionCurrentDateTime(divisionCode);
			dtHeader.Rows.Add(drHeader);
			//
#if DEBUG
            //dsHeader.WriteXml("E:\\KMan\\VBECS 2.0.0.1\\ReportHeader.xml", XmlWriteMode.WriteSchema);
#endif
			//
			return dsHeader;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5380"> 
		///		<ExpectedInput>Valid report criteria string</ExpectedInput>
		///		<ExpectedOutput>Data table containing report footer info</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5381"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Build DataSet used as a source for report footer
		/// </summary>
		/// <param name="reportCriteria">Report Criteria</param>
		/// <returns>Report Footer dataset</returns>
		public static DataSet GetReportFooter(string reportCriteria)
		{
			DataTable dtFooter = new DataTable("ReportFooter");
			DataSet dsFooter = new DataSet("VBECSReportFooter");
			dsFooter.Tables.Add(dtFooter);

			dtFooter.Columns.Add(ARTIFICIAL.ReportCriteria);

			DataRow drFooter = dtFooter.NewRow();
			drFooter[ARTIFICIAL.ReportCriteria] = reportCriteria;
			dtFooter.Rows.Add(drFooter);

			return dsFooter;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5382"> 
		///		<ExpectedInput>Valid printer name</ExpectedInput>
		///		<ExpectedOutput>Configured indicator of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5383"> 
		///		<ExpectedInput>Invalid printer name</ExpectedInput>
		///		<ExpectedOutput>Configured indicator of false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets value indicating whether the supplied printer is installed
		/// </summary>
		/// <param name="printerName">Printer Name</param>
		/// <returns>Printer configured indicator</returns>
		public static bool IsPrinterConfigured(string printerName)
		{
			System.Collections.ArrayList printers = BOL.VbecsReport.GetPrinters();
			//
			foreach(string printer in printers)
			{
				if (printerName == printer)
				{
					return true;
				}
			}
			//
			return false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5384"> 
		///		<ExpectedInput>Valid unsorted data table</ExpectedInput>
		///		<ExpectedOutput>Data table sorted to put user's division data first</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5385"> 
		///		<ExpectedInput>Data table without division code column</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sorts multi-divisional report information, to display user's division first
		/// BR_49.09
		/// </summary>
		/// <param name="dtUnsorted">Unsorted data table</param>
		/// <returns>Data table with user's division data first</returns>
		public static DataTable MultiDivisionalSort(DataTable dtUnsorted)
		{
			if (!dtUnsorted.Columns.Contains(TABLE.VamcDivision.DivisionCode)||
				!dtUnsorted.Columns.Contains(TABLE.VamcDivision.DivisionName))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.ColumnNotFoundInDataRow("division code").ResString);
			}
			if (!dtUnsorted.Columns.Contains(ARTIFICIAL.ThisDivision))
			{
				dtUnsorted.Columns.Add(ARTIFICIAL.ThisDivision, typeof(bool));
			}
			//
			foreach(DataRow dr in dtUnsorted.Rows)
			{
				dr[ARTIFICIAL.ThisDivision]=(dr[TABLE.VamcDivision.DivisionCode].ToString().Trim()==Common.LogonUser.LogonUserDivisionCode.Trim());
			}
			//
			DataRow[] drSort = dtUnsorted.Select("","ThisDivision DESC, DivisionName");
			//
			DataTable dtSorted = dtUnsorted.Clone();
            dtSorted.PrimaryKey = new DataColumn[0];

            foreach (DataColumn col in dtSorted.Columns)
            {
                col.AllowDBNull = true;
            }

			foreach(DataRow dr in drSort)
			{
				dtSorted.Rows.Add(dr.ItemArray);
			}
			//
			return dtSorted;
		}

        /// <summary>
        /// This method is used to enable/disable different buttons on the report
        /// Change done here will affect all reports
        /// </summary>
        /// <param name="rvReport"></param>
        public static void SetUpReportButtons(ReportViewer rvReport)
        {
            rvReport.ShowBackButton = false;
            rvReport.ShowContextMenu = false;
            rvReport.ShowExportButton = true;
            rvReport.ShowFindControls = true;
            rvReport.ShowParameterPrompts = false;
            rvReport.ShowPrintButton = true;
            rvReport.ShowRefreshButton = false;
            rvReport.ShowStopButton = false;
            rvReport.ShowZoomControl = true;
            rvReport.SetDisplayMode(DisplayMode.PrintLayout);
            rvReport.ZoomMode = ZoomMode.Percent;
            rvReport.ZoomPercent = 100;
            //CR 3384
            ToolStrip toolStrip = (ToolStrip)rvReport.Controls.Find("toolStrip1",true)[0];
            for (int i = 0; i < toolStrip.Items.Count; i++ )
            {
                if (toolStrip.Items[i].Name == "print" || toolStrip.Items[i].Name == "pageSetup")
                {
                    toolStrip.Items.RemoveAt(i);
                    i--;
                }
            }
        }
        
        /// <summary>
        /// Export report to pdf
        /// </summary>
        /// <param name="report"></param>
        /// <param name="fileName"></param>
        public static void ExportToPdf(LocalReport report, string fileName)
        {
            Warning[] warnings;
            string[] streamIds;
            string mimeType = string.Empty;
            string encoding = string.Empty;
            string extension = string.Empty;

            byte[] bytes = report.Render("PDF", null, out mimeType, out encoding, out extension, out streamIds,
                                         out warnings);

            System.IO.FileStream fileStream = new System.IO.FileStream(fileName, FileMode.Create, FileAccess.Write);
            fileStream.Write(bytes, 0, bytes.Length);
            fileStream.Close();
        }
		#endregion

		#region Methods

		/// <summary>
		/// Forms report range part of report criteria. May be used by subreports.
		/// </summary>
		/// <returns>String describing date range of the report.</returns>
		protected virtual string GetDateRangeReportCriteria()
		{
			return String.Concat( 
				StrRes.OtherMsg.Common.ReportsDateRangeCriteriaTitle().ResString, 
				Common.VBECSDateTime.FormatDateString( this.StartDate ),
				StrRes.OtherMsg.Common.ReportsDateRangeCriteriaSeparator().ResString,
				Common.VBECSDateTime.FormatDateString( this.EndDate ) );
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6669"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6670"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadPropertiesFromDataRow
		/// </summary>
		/// <param name="dr"></param>
		public void LoadPropertiesFromDataRow(DataRow dr)
		{
			this._parameterRow.ItemArray = dr.ItemArray;
		}

		#endregion

		#region Report Scheduler
		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6671"> 
		///		<ExpectedInput>Valid Vbecs Report object</ExpectedInput>
		///		<ExpectedOutput>Parameters saved to ScheduledReport table</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6672"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save parameters
		/// </summary>
		public void SaveParameters()
		{
			this.DivisionCode = Common.LogonUser.LogonUserDivisionCode;
			this.PrintedBy = Common.LogonUser.LogonUserName;
			this.ReportName = this.GetType().FullName;
			this.InsertScheduledReport();
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6673"> 
		///		<ExpectedInput>Valid Vbecs Report object</ExpectedInput>
		///		<ExpectedOutput>Parameters saved to ScheduledReport table</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6674"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Schedule At date/time
		/// </summary>
		public void Schedule()
		{
			SaveParameters();
		}
		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7322"> 
		///		<ExpectedInput>Valid task status</ExpectedInput>
		///		<ExpectedOutput>Data table of pending scheduled reports</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7323"> 
		///		<ExpectedInput>Invalid task status</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get all scheduled reports in a specified task status
		/// </summary>
		/// <param name="status">Task Status code</param>
		/// <returns>Scheduled Report data table</returns>
		public static DataTable GetScheduledReports(Common.TaskStatus status)
		{
			return DAL.VbecsReport.GetScheduledReports(status);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7324"> 
		///		<ExpectedInput>Valid scheduled report guid</ExpectedInput>
		///		<ExpectedOutput>Data table containing scheduled report details</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7325"> 
		///		<ExpectedInput>Invalid scheduled report guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get a scheduled report record
		/// </summary>
		/// <param name="scheduledReportGuid">Scheduled Report Guid</param>
		/// <returns>Scheduled Report data table</returns>
		public static DataTable GetScheduledReport(Guid scheduledReportGuid)
		{
			return DAL.VbecsReport.GetScheduledReport(scheduledReportGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7326"> 
		///		<ExpectedInput>Valid scheduled report guid and row version</ExpectedInput>
		///		<ExpectedOutput>Success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7327"> 
		///		<ExpectedInput>Invalid scheduled report guid and row version</ExpectedInput>
		///		<ExpectedOutput>Row version exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Update the task status of a scheduled report record
		/// </summary>
		/// <param name="scheduledReportGuid">Scheduled Report Guid</param>
		/// <param name="status">Task Status code</param>
		/// <param name="rowVersion">Row Version</param>
		/// <returns>Success indicator</returns>
		public static bool UpdateScheduledReportStatus(Guid scheduledReportGuid,Common.TaskStatus status, byte[] rowVersion)
		{
			return DAL.VbecsReport.UpdateScheduledReportStatus(scheduledReportGuid, status, rowVersion);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7328"> 
		///		<ExpectedInput>Valid scheduled report object</ExpectedInput>
		///		<ExpectedOutput>Success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7329"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create a scheduled report record
		/// </summary>
		/// <returns>Success indicator</returns>
		public bool InsertScheduledReport()
		{
			DateTime printDate = new DateTime(this.PrintDate.Year, this.PrintDate.Month, this.PrintDate.Day, this.PrintDate.Hour, this.PrintDate.Minute, 0);
			return DAL.VbecsReport.InsertScheduledReport(this._parametersDataSet,printDate);
		}

        /// <summary>
        /// Generate report and schedule for future print
        /// </summary>
        public void GenerateAndPrintReport()
        {
            using (var reportPrinter = createReportPrinter())
            {
                if (reportPrinter != null)
                    reportPrinter.Print();
            }
        }

        /// <summary>
        /// Will be derived by child classes
        /// </summary>
        /// <returns></returns>
        protected abstract VbecsReportPrinter createReportPrinter();

		#endregion
	}
}
